/**
 * Search module for finding financial instruments and related information.
 *
 * This module provides search functionality to find stocks, ETFs, mutual funds,
 * and other financial instruments by name, symbol, or keywords. It also returns
 * related news articles and other relevant information.
 *
 * @example Basic Search
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Search by ticker
 * const results = await yahooFinance.search('AAPL');
 * console.log(results.quotes[0]); // { symbol: 'AAPL', shortname: 'Apple Inc.', ... }
 *
 * // Search by company name
 * const google = await yahooFinance.search('Alphabet');
 * console.log(google.quotes); // [{ symbol: 'GOOGL', ... }, { symbol: 'GOOG', ... }]
 * ```
 *
 * @example Advanced Search Options
 * ```typescript
 * // Limit results and get news
 * const results = await yahooFinance.search('Tesla', {
 *   quotesCount: 5,
 *   newsCount: 10
 * });
 *
 * console.log(results.quotes.length); // Up to 5 quotes
 * console.log(results.news.length);   // Up to 10 news articles
 *
 * // Regional search
 * const ukResults = await yahooFinance.search('Vodafone', {
 *   region: 'GB',
 *   lang: 'en-GB'
 * });
 * ```
 *
 * @example Working with Results
 * ```typescript
 * const results = await yahooFinance.search('Microsoft');
 *
 * // Find exact symbol match
 * const msft = results.quotes.find(q => q.symbol === 'MSFT');
 * if (msft && 'shortname' in msft) {
 *   console.log(msft.shortname); // "Microsoft Corporation"
 * }
 *
 * // Get all equity symbols
 * const equities = results.quotes.filter(q =>
 *   'quoteType' in q && q.quoteType === 'EQUITY'
 * );
 *
 * // Browse news articles
 * results.news.forEach(article => {
 *   console.log(`${article.title} - ${article.publisher}`);
 * });
 *
 * // Check performance timing
 * console.log(`Search took ${results.totalTime}ms total`);
 * console.log(`Quotes: ${results.timeTakenForQuotes}ms`);
 * console.log(`News: ${results.timeTakenForNews}ms`);
 * ```
 *
 * ### Notes
 * 1. See {@link SearchOptions} for all available options, e.g.
 *     1. **Fuzzy Matching**: When enabled, allows approximate matching for typos
 *        and partial company names.
 *
 * 1. See {@link SearchResult} for complete result structure, comprising:
 *     1. **Yahoo Finance Symbols**: Stocks, ETFs, funds, etc. with trading data
 *     1. **Non-Yahoo Entities**: Companies / startups from Crunchbase (isYahooFinance: false)
 *     1. **News Articles**: Related financial news and analysis
 *     1. **Research reports**
 *     1. **Performance**: detailed timing information for different
 *     search components, useful for performance monitoring and optimization.
 *
 * @module search
 */
import type { ModuleOptionsWithValidateFalse, ModuleOptionsWithValidateTrue, ModuleThis } from "../lib/moduleCommon.js";
/**
 * Base interface for all Yahoo Finance search quote results.
 */
export interface SearchQuoteYahoo {
    [key: string]: unknown;
    /** Trading symbol */
    symbol: string;
    /** Whether this result is from Yahoo Finance */
    isYahooFinance: true;
    /** Stock exchange code */
    exchange: string;
    /** Exchange display name */
    exchDisp?: string;
    /** Short display name */
    shortname?: string;
    /** Full company/instrument name */
    longname?: string;
    /** Search result index identifier */
    index: "quotes";
    /** Search relevance score */
    score: number;
    /** New listing date for recent IPOs */
    newListingDate?: Date;
    /** Previous name before name change */
    prevName?: string;
    /** Date of name change */
    nameChangeDate?: Date;
    /** Business sector */
    sector?: string;
    industry?: string;
    dispSecIndFlag?: boolean;
}
export interface SearchQuoteYahooEquity extends SearchQuoteYahoo {
    quoteType: "EQUITY";
    typeDisp: "Equity";
    sectorDisp?: string;
    industryDisp?: string;
}
export interface SearchQuoteYahooOption extends SearchQuoteYahoo {
    quoteType: "OPTION";
    typeDisp: "Option";
}
export interface SearchQuoteYahooETF extends SearchQuoteYahoo {
    quoteType: "ETF";
    typeDisp: "ETF";
}
export interface SearchQuoteYahooFund extends SearchQuoteYahoo {
    quoteType: "MUTUALFUND";
    typeDisp: "Fund";
}
export interface SearchQuoteYahooIndex extends SearchQuoteYahoo {
    quoteType: "INDEX";
    typeDisp: "Index";
}
export interface SearchQuoteYahooCurrency extends SearchQuoteYahoo {
    quoteType: "CURRENCY";
    typeDisp: "Currency";
}
export interface SearchQuoteYahooCryptocurrency extends SearchQuoteYahoo {
    quoteType: "CRYPTOCURRENCY";
    typeDisp: "Cryptocurrency";
}
export interface SearchQuoteYahooFuture extends SearchQuoteYahoo {
    quoteType: "FUTURE";
    typeDisp: "Future" | "Futures";
}
export interface SearchQuoteYahooMoneyMarket extends SearchQuoteYahoo {
    quoteType: "MONEY_MARKET";
    typeDisp: "MoneyMarket";
}
export interface SearchQuoteNonYahoo {
    [key: string]: unknown;
    index: string;
    name: string;
    permalink: string;
    isYahooFinance: false;
}
export interface SearchNews {
    [key: string]: unknown;
    uuid: string;
    title: string;
    publisher: string;
    link: string;
    providerPublishTime: Date;
    type: string;
    thumbnail?: {
        resolutions: SearchNewsThumbnailResolution[];
    };
    relatedTickers?: string[];
}
export interface SearchNewsThumbnailResolution {
    url: string;
    width: number;
    height: number;
    tag: string;
}
/**
 * Complete search result containing quotes, news, and other information.
 */
export interface SearchResult {
    [key: string]: unknown;
    /** Explanatory information (usually empty) */
    explains: Array<unknown>;
    /** Total number of quote results found */
    count: number;
    /** Array of financial instrument search results */
    quotes: Array<SearchQuoteYahooEquity | SearchQuoteYahooOption | SearchQuoteYahooETF | SearchQuoteYahooFund | SearchQuoteYahooIndex | SearchQuoteYahooCurrency | SearchQuoteYahooCryptocurrency | SearchQuoteNonYahoo | SearchQuoteYahooFuture | SearchQuoteYahooMoneyMarket>;
    /** Array of related news articles */
    news: Array<SearchNews>;
    /** Navigation results (structure TBD) */
    nav: Array<unknown>;
    /** List results (structure TBD) */
    lists: Array<unknown>;
    /** Research report results (structure TBD) */
    researchReports: Array<unknown>;
    /** Total time taken for the search request in milliseconds */
    totalTime: number;
    /**
     * Screener field results (structure TBD)
     * @remarks Temporarily optional due to recent API addition
     */
    screenerFieldResults?: Array<unknown>;
    /**
     * Cultural assets results (structure TBD)
     * @remarks Temporarily optional due to recent API addition
     */
    culturalAssets?: Array<unknown>;
    /** Time taken for quote search in milliseconds */
    timeTakenForQuotes: number;
    /** Time taken for news search in milliseconds */
    timeTakenForNews: number;
    /** Time taken for algo watchlist in milliseconds */
    timeTakenForAlgowatchlist: number;
    /** Time taken for predefined screener in milliseconds */
    timeTakenForPredefinedScreener: number;
    /** Time taken for Crunchbase search in milliseconds */
    timeTakenForCrunchbase: number;
    /** Time taken for navigation search in milliseconds */
    timeTakenForNav: number;
    /** Time taken for research reports in milliseconds */
    timeTakenForResearchReports: number;
    /** Time taken for screener field search in milliseconds */
    timeTakenForScreenerField: number;
    /** Time taken for cultural assets search in milliseconds */
    timeTakenForCulturalAssets: number;
    /** Time taken for search lists in milliseconds */
    timeTakenForSearchLists: number;
}
/**
 * Configuration options for search requests.
 */
export interface SearchOptions {
    /** Language code for search results (e.g., "en-US") */
    lang?: string;
    /** Region code for search results (e.g., "US") */
    region?: string;
    /** Maximum number of quote results to return */
    quotesCount?: number;
    /** Maximum number of news results to return */
    newsCount?: number;
    /** Whether to enable fuzzy matching for search terms */
    enableFuzzyQuery?: boolean;
    /** Query ID for quotes search algorithm */
    quotesQueryId?: string;
    /** Query ID for multi-quote search algorithm */
    multiQuoteQueryId?: string;
    /** Query ID for news search algorithm */
    newsQueryId?: string;
    /** Whether to enable Crunchbase results */
    enableCb?: boolean;
    /** Whether to enable navigation links */
    enableNavLinks?: boolean;
    /** Whether to enable enhanced trivial query processing */
    enableEnhancedTrivialQuery?: boolean;
}
/**
 * Search for financial instruments with validation enabled.
 *
 * **See the {@link [modules/search] search module} docs for examples and more.**
 * @see {@link [modules/search] search module} docs for examples and more.
 *
 * @param query - Search term (symbol, company name, or keywords)
 * @param queryOptionsOverrides - Optional search configuration
 * @param moduleOptions - Optional module configuration
 * @returns Promise resolving to validated SearchResult
 */
export default function search(this: ModuleThis, query: string, queryOptionsOverrides?: SearchOptions, moduleOptions?: ModuleOptionsWithValidateTrue): Promise<SearchResult>;
/**
 * Search for financial instruments with validation disabled.
 *
 * **See the {@link [modules/search] search module} docs for examples and more.**
 * @see {@link [modules/search] search module} docs for examples and more.
 *
 * @param query - Search term (symbol, company name, or keywords)
 * @param queryOptionsOverrides - Optional search configuration
 * @param moduleOptions - Module configuration with validateResult: false
 * @returns Promise resolving to unvalidated raw data
 */
export default function search(this: ModuleThis, query: string, queryOptionsOverrides?: SearchOptions, moduleOptions?: ModuleOptionsWithValidateFalse): Promise<unknown>;
//# sourceMappingURL=search.d.ts.map