/**
 * Quote Summary module for retrieving comprehensive financial data.
 *
 * This module provides detailed financial information for symbols including
 * price data, company profile, financial statements, recommendations, and more.
 * It's modular - you can request only the specific data modules you need.
 *
 * @example Basic Usage
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Default modules: price + summaryDetail
 * const result = await yahooFinance.quoteSummary('AAPL');
 * console.log(result.price?.regularMarketPrice);
 * console.log(result.summaryDetail?.marketCap);
 *
 * // Request specific modules
 * const profile = await yahooFinance.quoteSummary('AAPL', {
 *   modules: ['summaryProfile', 'financialData']
 * });
 * console.log(profile.summaryProfile?.longBusinessSummary);
 *
 * // Get all available modules
 * const comprehensive = await yahooFinance.quoteSummary('AAPL', {
 *   modules: 'all'
 * });
 * ```
 *
 * ### Available Modules
 *
 * See {@linkcode quoteSummary_modules} for the full list.
 *
 * ### Notes
 * * In the original **node-yahoo-finance v1**, this was incorrectly called "quote".
 *   The actual quote data is now in the {@linkcode [modules/quote] quote} module.
 *   This module provides much more comprehensive data than basic quotes.
 *
 * * **Performance**: requesting fewer modules improves response time.
 * Only request the modules you actually need.
 *
 * * **Financial Statements**: Since Nov 2024, the ~~`incomeStatementHistory*`~~,
 *   ~~`balanceSheetHistory*`~~, ~~`cashflowStatementHistory*`~~ modules provide
 *   almost no data.  Use the
 *   {@linkcode [modules/fundamentalsTimeSeries] fundamentalsTimeSeries}
 *   module instead.
 *
 * @module quoteSummary
 */
import type { QuoteSummaryResult } from "./quoteSummary-iface.js";
import type { ModuleOptionsWithValidateFalse, ModuleOptionsWithValidateTrue, ModuleThis } from "../lib/moduleCommon.js";
export type * from "./quoteSummary-iface.js";
/**
 * Available quote summary modules for requesting specific data.
 *
 * Each module provides different types of financial data:
 * - **Company Info**: `assetProfile`, `summaryProfile`, `quoteType`
 * - **Price Data**: `price`, `summaryDetail`
 * - **Financial Statements**: ~~`incomeStatementHistory*`~~,
 *   ~~`balanceSheetHistory*`~~, ~~`cashflowStatementHistory*`~~. *Use
 *   {@linkcode [modules/fundamentalsTimeSeries] fundamentalsTimeSeries} instead!*
 * - **Analysis**: `recommendationTrend`, `upgradeDowngradeHistory`, `earnings*`
 * - **Ownership**: `institutionOwnership`, `fundOwnership`, `majorHoldersBreakdown`
 * - **Fund Specific**: `fundProfile`, `fundPerformance`, `topHoldings`
 * - **Other**: `calendarEvents`, `defaultKeyStatistics`, `financialData`
 *
 * Modules ending with "*" have both regular and quarterly versions.
 */
export declare const quoteSummary_modules: string[];
/**
 * Type union of all available quote summary module names.
 * Use this to specify which modules to include in your request.
 */
export type QuoteSummaryModules = "assetProfile" | "balanceSheetHistory" | "balanceSheetHistoryQuarterly" | "calendarEvents" | "cashflowStatementHistory" | "cashflowStatementHistoryQuarterly" | "defaultKeyStatistics" | "earnings" | "earningsHistory" | "earningsTrend" | "financialData" | "fundOwnership" | "fundPerformance" | "fundProfile" | "incomeStatementHistory" | "incomeStatementHistoryQuarterly" | "indexTrend" | "industryTrend" | "insiderHolders" | "insiderTransactions" | "institutionOwnership" | "majorDirectHolders" | "majorHoldersBreakdown" | "netSharePurchaseActivity" | "price" | "quoteType" | "recommendationTrend" | "secFilings" | "sectorTrend" | "summaryDetail" | "summaryProfile" | "topHoldings" | "upgradeDowngradeHistory";
/**
 * Configuration options for quote summary requests.
 */
export interface QuoteSummaryOptions {
    /**
     * Whether to return formatted strings instead of raw numbers.
     * When true, numbers like 1000000 become "1.00M".
     * @defaultValue false
     */
    formatted?: boolean;
    /**
     * Which data modules to include in the response.
     * - Array of specific module names for targeted data
     * - "all" to get all available modules
     * @defaultValue ["price", "summaryDetail"]
     */
    modules?: Array<QuoteSummaryModules> | "all";
}
/**
 * Get comprehensive quote summary data with validation enabled.
 *
 * **See the {@link [modules/quoteSummary] quoteSummary module} docs for examples and more.**
 * @see {@link [modules/quoteSummary] quoteSummary module} docs for examples and more.
 *
 * @param symbol - Stock symbol to get data for
 * @param queryOptionsOverrides - Optional configuration for modules and formatting
 * @param moduleOptions - Optional module configuration
 * @returns Promise resolving to validated QuoteSummaryResult
 */
export default function quoteSummary(this: ModuleThis, symbol: string, queryOptionsOverrides?: QuoteSummaryOptions, moduleOptions?: ModuleOptionsWithValidateTrue): Promise<QuoteSummaryResult>;
/**
 * Get comprehensive quote summary data with validation disabled.
 *
 * **See the {@link [modules/quoteSummary] quoteSummary module} docs for examples and more.**
 * @see {@link [modules/quoteSummary] quoteSummary module} docs for examples and more.
 *
 * @param symbol - Stock symbol to get data for
 * @param queryOptionsOverrides - Optional configuration for modules and formatting
 * @param moduleOptions - Module configuration with validateResult: false
 * @returns Promise resolving to unvalidated raw data, but resembling {@linkcode QuoteSummaryResult}.
 */
export default function quoteSummary(this: ModuleThis, symbol: string, queryOptionsOverrides?: QuoteSummaryOptions, moduleOptions?: ModuleOptionsWithValidateFalse): Promise<unknown>;
//# sourceMappingURL=quoteSummary.d.ts.map