"use strict";
/**
 * Analyst insights, research reports, and company analysis.
 *
 * The module retrieves comprehensive analyst coverage including recommendations,
 * research reports, significant developments, and company analysis scores.
 *
 * @example Basic Usage
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Get insights for a symbol
 * const insights = await yahooFinance.insights('AAPL');
 * console.log(insights.recommendation); // { rating: "BUY", targetPrice: 200, ... }
 * console.log(insights.sigDevs[0]); // Recent significant development
 * ```
 *
 * @example Working with Reports
 * ```typescript
 * const insights = await yahooFinance.insights('TSLA');
 *
 * // Browse research reports
 * insights.reports?.forEach(report => {
 *   console.log(`${report.reportTitle} by ${report.provider}`);
 *   console.log(`Date: ${report.reportDate}`);
 * });
 *
 * // Check significant developments
 * insights.sigDevs.forEach(dev => {
 *   console.log(`${dev.date}: ${dev.headline}`);
 * });
 * ```
 *
 * @example Research Reports
 * ```typescript
 * const insights = await yahooFinance.insights('TSLA', {
 *   reportsCount: 5
 * });
 *
 * // Browse research reports
 * insights.reports?.forEach(report => {
 *   console.log(`${report.reportTitle} - ${report.provider}`);
 *   console.log(`Published: ${report.reportDate}`);
 *   console.log(`Rating: ${report.investmentRating}`);
 *   if (report.targetPrice) {
 *     console.log(`Target: $${report.targetPrice}`);
 *   }
 * });
 * ```
 *
 * @example Company Analysis
 * ```typescript
 * const insights = await yahooFinance.insights('MSFT');
 *
 * // Company performance vs sector
 * if (insights.companySnapshot) {
 *   const company = insights.companySnapshot.company;
 *   const sector = insights.companySnapshot.sector;
 *
 *   console.log('Company vs Sector Scores:');
 *   console.log(`Innovation: ${company.innovativeness} vs ${sector.innovativeness}`);
 *   console.log(`Hiring: ${company.hiring} vs ${sector.hiring}`);
 *   console.log(`Sustainability: ${company.sustainability} vs ${sector.sustainability}`);
 * }
 * ```
 *
 * @remarks
 * **Data Availability**: Insights data varies significantly by symbol.
 * Popular large-cap stocks typically have the most comprehensive coverage.
 *
 * **Premium Content**: Some research reports and detailed analysis may
 * require premium subscriptions or additional access rights.
 *
 * **Timeliness**: Recommendations and reports reflect the most recent
 * analyst coverage but may not include the very latest market developments.
 *
 * @see {@link InsightsOptions} for all available options
 * @see {@link InsightsResult} for complete result structure
 *
 * @remarks
 * **Data Availability**: Insights data availability varies by symbol.
 * Premium content may require additional subscriptions.
 *
 * @module insights
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = insights;
const index_js_1 = require("../lib/validate/index.js");
// @yf-schema: see the docs on how this file is automatically updated.
// @ts-ignore: tmp
const insights_schema_js_1 = __importDefault(require("./insights.schema.js"));
const definitions = (0, index_js_1.getTypedDefinitions)(insights_schema_js_1.default);
const queryOptionsDefaults = {
    lang: "en-US",
    region: "US",
    getAllResearchReports: true,
    reportsCount: 2,
};
/*
 * @param symbol - Stock symbol to get insights for.  Use search() to find valid symbols.
 * @param queryOptionsOverrides - Optional configuration:
 *                                - `reportsCount`: Number of research reports (default: 2)
 *                                - `region`: Market region (default: "US")
 *                                - `lang`: Language for results (default: "en-US")
 * @param moduleOptions - Optional module configuration (validateResult, etc.)
 *
 * @returns Promise that resolves to an InsightsResult containing:
 *          - `recommendation`: Current analyst recommendation and target price
 *          - `sigDevs`: Array of significant developments and news
 *          - `reports`: Array of research reports with ratings and analysis
 *          - `companySnapshot`: Company performance scores vs sector/market
 *          - `events`: Upcoming events and earnings
 *
 * @throws Will throw an error if:
 *         - Network request fails
 *         - Invalid symbol
 *         - Validation fails (if enabled)
 */
function insights(symbol, queryOptionsOverrides, moduleOptions) {
    return this._moduleExec({
        moduleName: "insights",
        query: {
            assertSymbol: symbol,
            url: "https://${YF_QUERY_HOST}/ws/insights/v2/finance/insights",
            definitions,
            schemaKey: "#/definitions/InsightsOptions",
            defaults: queryOptionsDefaults,
            overrides: queryOptionsOverrides,
            runtime: { symbol },
        },
        result: {
            definitions,
            schemaKey: "#/definitions/InsightsResult",
            transformWith(result) {
                if (!result.finance) {
                    throw new Error("Unexpected result: " + JSON.stringify(result));
                }
                return result.finance.result;
            },
        },
        moduleOptions,
    });
}
