/**
 * Historical data module for retrieving price history, dividends, and stock splits.
 *
 * This module provides historical price data, dividend payments, and stock split
 * information for financial instruments. While functional, many users prefer
 * the {@link chart} module which offers more flexibility and features.
 *
 * @example Basic Price History
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Get 1 year of daily data
 * const history = await yahooFinance.historical('AAPL', {
 *   period1: '2023-01-01',
 *   period2: '2024-01-01'
 * });
 *
 * console.log(history[0]); // Most recent day
 * // { date: Date, open: 150.5, high: 155.2, low: 149.8, close: 154.1, ... }
 * ```
 *
 * @example Different Intervals
 * ```typescript
 * // Weekly data
 * const weekly = await yahooFinance.historical('AAPL', {
 *   period1: '2023-01-01',
 *   interval: '1wk'
 * });
 *
 * // Monthly data
 * const monthly = await yahooFinance.historical('AAPL', {
 *   period1: '2022-01-01',
 *   interval: '1mo'
 * });
 * ```
 *
 * @example Dividends and Splits
 * ```typescript
 * // Get dividend history
 * const dividends = await yahooFinance.historical('AAPL', {
 *   period1: '2023-01-01',
 *   events: 'dividends'
 * });
 * // Returns: [{ date: Date, dividends: 0.24 }, ...]
 *
 * // Get stock splits
 * const splits = await yahooFinance.historical('AAPL', {
 *   period1: '2020-01-01',
 *   events: 'split'
 * });
 * // Returns: [{ date: Date, stockSplits: "4:1" }, ...]
 * ```
 *
 * @remarks
 * **Limitations**:
 * - Intervals limited to daily ("1d"), weekly ("1wk"), monthly ("1mo")
 * - Events (prices, dividends, splits) require separate requests
 * - Consider using {@link chart} module for more flexibility
 *
 * **Performance**: The chart module often provides better performance and
 * more features for historical data needs.
 *
 * @module historical
 */
import type { ModuleOptionsWithValidateFalse, ModuleOptionsWithValidateTrue, ModuleThis } from "../lib/moduleCommon.js";
/** Array of historical price data points */
export type HistoricalHistoryResult = Array<HistoricalRowHistory>;
/** Array of dividend payment records */
export type HistoricalDividendsResult = Array<HistoricalRowDividend>;
/** Array of stock split records */
export type HistoricalStockSplitsResult = Array<HistoricalRowStockSplit>;
/** Union type for all possible historical data results */
export type HistoricalResult = HistoricalHistoryResult | HistoricalDividendsResult | HistoricalStockSplitsResult;
/**
 * Historical price data for a single trading day.
 */
export interface HistoricalRowHistory {
    [key: string]: unknown;
    /** Trading date */
    date: Date;
    /** Opening price */
    open: number;
    /** Highest price during the day */
    high: number;
    /** Lowest price during the day */
    low: number;
    /** Closing price */
    close: number;
    /** Adjusted closing price (accounts for splits/dividends) */
    adjClose?: number;
    /** Trading volume */
    volume: number;
}
/**
 * Dividend payment record.
 */
export interface HistoricalRowDividend {
    /** Ex-dividend date */
    date: Date;
    /** Dividend amount per share */
    dividends: number;
}
/**
 * Stock split record.
 */
export interface HistoricalRowStockSplit {
    /** Split effective date */
    date: Date;
    /** Split ratio as string (e.g., "2:1", "4:1") */
    stockSplits: string;
}
/**
 * Configuration options for historical data requests.
 */
export interface HistoricalOptions {
    /**
     * Start date for historical data.
     * Can be Date object, ISO string, or Unix timestamp.
     */
    period1: Date | string | number;
    /**
     * End date for historical data (defaults to current date).
     * Can be Date object, ISO string, or Unix timestamp.
     */
    period2?: Date | string | number;
    /**
     * Data interval/frequency.
     * @defaultValue "1d"
     */
    interval?: "1d" | "1wk" | "1mo";
    /**
     * Type of historical data to retrieve.
     * @defaultValue "history"
     */
    events?: "history" | "dividends" | "split";
    /**
     * Whether to include adjusted close prices.
     * @defaultValue true
     */
    includeAdjustedClose?: boolean;
}
/** Historical options specifically for price history */
export interface HistoricalOptionsEventsHistory extends HistoricalOptions {
    events?: "history";
}
/** Historical options specifically for dividend data */
export interface HistoricalOptionsEventsDividends extends HistoricalOptions {
    events: "dividends";
}
/** Historical options specifically for stock split data */
export interface HistoricalOptionsEventsSplit extends HistoricalOptions {
    events: "split";
}
/**
 * Get historical price data for a symbol.
 *
 * **See the {@link [modules/historical] historical module} docs for examples and more.**
 * @see {@link [modules/historical] historical module} docs for examples and more.
 *
 * @param symbol - Stock symbol to get historical data for
 * @param queryOptionsOverrides - Configuration for price history
 * @param moduleOptions - Optional module configuration
 * @returns Promise resolving to array of historical price records
 */
export default function historical(this: ModuleThis, symbol: string, queryOptionsOverrides: HistoricalOptionsEventsHistory, moduleOptions?: ModuleOptionsWithValidateTrue): Promise<HistoricalHistoryResult>;
/**
 * Get dividend history for a symbol.
 *
 * **See the {@link [modules/historical] historical module} docs for examples and more.**
 * @see {@link [modules/historical] historical module} docs for examples and more.
 *
 * @param symbol - Stock symbol to get dividend data for
 * @param queryOptionsOverrides - Configuration with events: "dividends"
 * @param moduleOptions - Optional module configuration
 * @returns Promise resolving to array of dividend records
 */
export default function historical(this: ModuleThis, symbol: string, queryOptionsOverrides: HistoricalOptionsEventsDividends, moduleOptions?: ModuleOptionsWithValidateTrue): Promise<HistoricalDividendsResult>;
/**
 * Get stock split history for a symbol.
 *
 * **See the {@link [modules/historical] historical module} docs for examples and more.**
 * @see {@link [modules/historical] historical module} docs for examples and more.
 *
 * @param symbol - Stock symbol to get split data for
 * @param queryOptionsOverrides - Configuration with events: "split"
 * @param moduleOptions - Optional module configuration
 * @returns Promise resolving to array of stock split records
 */
export default function historical(this: ModuleThis, symbol: string, queryOptionsOverrides: HistoricalOptionsEventsSplit, moduleOptions?: ModuleOptionsWithValidateTrue): Promise<HistoricalStockSplitsResult>;
/**
 * Get historical data with validation disabled.
 *
 * **See the {@link [modules/historical] historical module} docs for examples and more.**
 * @see {@link [modules/historical] historical module} docs for examples and more.
 *
 * @param symbol - Stock symbol to get data for
 * @param queryOptionsOverrides - Configuration options
 * @param moduleOptions - Module configuration with validateResult: false
 * @returns Promise resolving to unvalidated raw data
 */
export default function historical(this: ModuleThis, symbol: string, queryOptionsOverrides: HistoricalOptions, moduleOptions?: ModuleOptionsWithValidateFalse): Promise<any>;
//# sourceMappingURL=historical.d.ts.map