import { ExtendedCookieJar } from "../cookieJar.js";
import type { YahooFinance } from "../../createYahooFinance.js";
import { type Logger } from "./logger.js";
import type { ValidationOptions } from "../validateAndCoerceTypes.js";
import type { QueueOptions } from "../queue.js";
import type { NOTICE_IDS } from "../notices.js";
import type { QuoteCombineOptions } from "../../other/quoteCombine.js";
/**
 * Options for {@linkcode YahooFinance}.
 *
 * @example
 * ```ts
 * import YahooFinance from 'yahoo-finance2';
 * const yahooFinance = new YahooFinance({
 *   suppressNotices: ["yahooSurvey"],
 *   // etc
 * });
 * ```
 */
export interface YahooFinanceOptions {
    /**
     * Where to send queries.  Default: `query2.finance.yahoo.com`.
     *
     * As per
     * [this stackoverflow post](https://stackoverflow.com/questions/44030983/yahoo-finance-url-not-working/47505102#47505102):
     *
     * - `query1.finance.yahoo.com` serves `HTTP/1.0`
     * - `query2.finance.yahoo.com` serves `HTTP/1.1`
     * - [Differences between HTTP/1.0 and HTTP/1.1](https://stackoverflow.com/questions/246859/http-1-0-vs-1-1)
     *
     * Note: this does not affect redirects to other hosts used by e.g. Yahoo's cookies and consent.
     */
    YF_QUERY_HOST?: string;
    /** Override the default queue options, e.g. concurrency and timeout. */
    queue?: QueueOptions;
    /** Override the default validation options, e.g. logErrors, logOptionsErrors, etc.  */
    validation?: ValidationOptions;
    /** Optional array of notice ids to suppress, e.g. ["yahooSurvey"] */
    suppressNotices?: NOTICE_IDS[];
    /** Override the default quote combine options, e.g. maxSymbolsPerRequest, debounceTime. */
    quoteCombine?: QuoteCombineOptions;
    /** On errors, check if we're using the latest version and notify otherwise (default: true) */
    versionCheck?: boolean;
    /**
     * By default, we use an in-memory cookie store to re-use Yahoo cookies across requests.
     * This is usually fine for long running servers, but with serverless / edge functions
     * for example - since the initial cookie retrieval takes longer - you can speed up future
     * requests by providing a custom cookie jar with a database/redis backend.  For the CLI, we
     * likewise use a filesystem-backed cookie jar for this purpose.  See
     * {@linkcode ExtendedCookieJar} for more details and examples (based on
     * {@link https://www.npmjs.com/package/tough-cookie|npm:tough-cookie}).
     */
    cookieJar?: ExtendedCookieJar;
    /**
     * By default, we use the built-in `console` for logging, but you can override it with
     * anything you like.  You can use this to control logging output or send your logs to
     * a logging service.  See
     * {@linkcode Logger} for more details and examples.
     */
    logger?: Logger;
    /**
     * By default, we'll use `globalThis.fetch` at call time for HTTP requests, however,
     * you can override it with a custom fetch implementation.  You can also override
     * `fetch` per request with {@linkcode ModuleOptions}.
     */
    fetch?: typeof fetch;
    /**
     * Any options to pass to `fetch()` for all requests.  You can also override
     * `fetchOptions` per request with {@linkcode ModuleOptions}.
     */
    fetchOptions?: RequestInit;
}
type Obj = Record<string, unknown>;
export declare function mergeObjects(original: Obj, objToMerge: Obj): void;
export declare function validateOptions(this: YahooFinance, options: YahooFinanceOptions, source?: string): void;
export declare function setOptions(this: YahooFinance, options: YahooFinanceOptions): void;
export type { ExtendedCookieJar, Logger, NOTICE_IDS, QueueOptions, QuoteCombineOptions, ValidationOptions, };
export type { ModuleOptions, YahooFinanceFetchModuleOptions, } from "../moduleCommon.js";
//# sourceMappingURL=options.d.ts.map