/**
 * Trending Symbols module for retrieving symbols that are currently trending in a specific region.
 *
 * This module provides information about financial instruments that are currently
 * popular or receiving increased attention in a particular geographic region.
 *
 * @example Basic Usage
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Get trending symbols in the US
 * const usTrending = await yahooFinance.trendingSymbols('US');
 * console.log(usTrending.quotes[0]); // { symbol: 'TSLA' }
 *
 * // Get trending symbols in other regions
 * const ukTrending = await yahooFinance.trendingSymbols('GB');
 * const jpTrending = await yahooFinance.trendingSymbols('JP');
 * ```
 *
 * @example Customizing Results
 * ```typescript
 * // Get more trending symbols
 * const moreTrending = await yahooFinance.trendingSymbols('US', {
 *   count: 10
 * });
 *
 * // Different language/region combination
 * const germanTrending = await yahooFinance.trendingSymbols('DE', {
 *   lang: 'de-DE',
 *   region: 'DE'
 * });
 * ```
 *
 * @example Working with Results
 * ```typescript
 * const trending = await yahooFinance.trendingSymbols('US');
 *
 * console.log(`Found ${trending.count} trending symbols`);
 * trending.quotes.forEach((symbol, index) => {
 *   console.log(`${index + 1}. ${symbol.symbol}`);
 * });
 *
 * // Get full quotes for trending symbols
 * const symbols = trending.quotes.map(q => q.symbol);
 * const fullQuotes = await yahooFinance.quote(symbols);
 * ```
 *
 * ## Notes
 *
 * - **Data Freshness**: Trending data is updated regularly throughout trading
 * hours based on search volume, social media mentions, and trading activity.
 *
 * - **Regional Markets**: Each region shows symbols that are trending within
 * that specific market. US trending symbols may be very different from
 * Japanese or European trending symbols.
 *
 * - **Usage with other Modules**: The trending symbols can be used as input
 * to other modules like {@link quote} or {@link quoteSummary} to get detailed
 * information about the trending instruments.
 *
 * See {@link TrendingSymbolsOptions} for all available options.
 *
 * See {@link TrendingSymbolsResult} for complete result structure.
 *
 * @module trendingSymbols
 */
import { getTypedDefinitions } from "../lib/validate/index.js";
// @yf-schema: see the docs on how this file is automatically updated.
import schema from "./trendingSymbols.schema.js";
const definitions = getTypedDefinitions(schema);
const queryOptionsDefaults = {
    lang: "en-US",
    count: 5,
};
/**
 * @throws Will throw an error if:
 *         - Network request fails
 *         - Invalid region code
 *         - Validation fails (if enabled)
 */
export default function trendingSymbols(query, queryOptionsOverrides, moduleOptions) {
    return this._moduleExec({
        moduleName: "trendingSymbols",
        query: {
            url: "https://${YF_QUERY_HOST}/v1/finance/trending/" + query,
            definitions,
            schemaKey: "#/definitions/TrendingSymbolsOptions",
            defaults: queryOptionsDefaults,
            overrides: queryOptionsOverrides,
        },
        result: {
            definitions,
            schemaKey: "#/definitions/TrendingSymbolsResult",
            // deno-lint-ignore no-explicit-any
            transformWith(result) {
                if (!result.finance) {
                    throw new Error("Unexpected result: " + JSON.stringify(result));
                }
                return result.finance.result[0];
            },
        },
        moduleOptions,
    });
}
