/**
 * Search module for finding financial instruments and related information.
 *
 * This module provides search functionality to find stocks, ETFs, mutual funds,
 * and other financial instruments by name, symbol, or keywords. It also returns
 * related news articles and other relevant information.
 *
 * @example Basic Search
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Search by ticker
 * const results = await yahooFinance.search('AAPL');
 * console.log(results.quotes[0]); // { symbol: 'AAPL', shortname: 'Apple Inc.', ... }
 *
 * // Search by company name
 * const google = await yahooFinance.search('Alphabet');
 * console.log(google.quotes); // [{ symbol: 'GOOGL', ... }, { symbol: 'GOOG', ... }]
 * ```
 *
 * @example Advanced Search Options
 * ```typescript
 * // Limit results and get news
 * const results = await yahooFinance.search('Tesla', {
 *   quotesCount: 5,
 *   newsCount: 10
 * });
 *
 * console.log(results.quotes.length); // Up to 5 quotes
 * console.log(results.news.length);   // Up to 10 news articles
 *
 * // Regional search
 * const ukResults = await yahooFinance.search('Vodafone', {
 *   region: 'GB',
 *   lang: 'en-GB'
 * });
 * ```
 *
 * @example Working with Results
 * ```typescript
 * const results = await yahooFinance.search('Microsoft');
 *
 * // Find exact symbol match
 * const msft = results.quotes.find(q => q.symbol === 'MSFT');
 * if (msft && 'shortname' in msft) {
 *   console.log(msft.shortname); // "Microsoft Corporation"
 * }
 *
 * // Get all equity symbols
 * const equities = results.quotes.filter(q =>
 *   'quoteType' in q && q.quoteType === 'EQUITY'
 * );
 *
 * // Browse news articles
 * results.news.forEach(article => {
 *   console.log(`${article.title} - ${article.publisher}`);
 * });
 *
 * // Check performance timing
 * console.log(`Search took ${results.totalTime}ms total`);
 * console.log(`Quotes: ${results.timeTakenForQuotes}ms`);
 * console.log(`News: ${results.timeTakenForNews}ms`);
 * ```
 *
 * ### Notes
 * 1. See {@link SearchOptions} for all available options, e.g.
 *     1. **Fuzzy Matching**: When enabled, allows approximate matching for typos
 *        and partial company names.
 *
 * 1. See {@link SearchResult} for complete result structure, comprising:
 *     1. **Yahoo Finance Symbols**: Stocks, ETFs, funds, etc. with trading data
 *     1. **Non-Yahoo Entities**: Companies / startups from Crunchbase (isYahooFinance: false)
 *     1. **News Articles**: Related financial news and analysis
 *     1. **Research reports**
 *     1. **Performance**: detailed timing information for different
 *     search components, useful for performance monitoring and optimization.
 *
 * @module search
 */
import { getTypedDefinitions } from "../lib/validate/index.js";
// @yf-schema: see the docs on how this file is automatically updated.
import schema from "./search.schema.js";
const definitions = getTypedDefinitions(schema);
const queryOptionsDefaults = {
    lang: "en-US",
    region: "US",
    quotesCount: 6,
    newsCount: 4,
    enableFuzzyQuery: false,
    quotesQueryId: "tss_match_phrase_query",
    multiQuoteQueryId: "multi_quote_single_token_query",
    newsQueryId: "news_cie_vespa",
    enableCb: true,
    enableNavLinks: true,
    enableEnhancedTrivialQuery: true,
};
export default function search(query, queryOptionsOverrides, moduleOptions) {
    return this._moduleExec({
        moduleName: "search",
        query: {
            url: "https://${YF_QUERY_HOST}/v1/finance/search",
            definitions,
            schemaKey: "#/definitions/SearchOptions",
            defaults: queryOptionsDefaults,
            runtime: { q: query },
            overrides: queryOptionsOverrides,
        },
        result: {
            definitions,
            schemaKey: "#/definitions/SearchResult",
        },
        moduleOptions,
    });
}
