/**
 * Quote Summary module for retrieving comprehensive financial data.
 *
 * This module provides detailed financial information for symbols including
 * price data, company profile, financial statements, recommendations, and more.
 * It's modular - you can request only the specific data modules you need.
 *
 * @example Basic Usage
 * ```typescript
 * import YahooFinance from "yahoo-finance2";
 * const yahooFinance = new YahooFinance();
 *
 * // Default modules: price + summaryDetail
 * const result = await yahooFinance.quoteSummary('AAPL');
 * console.log(result.price?.regularMarketPrice);
 * console.log(result.summaryDetail?.marketCap);
 *
 * // Request specific modules
 * const profile = await yahooFinance.quoteSummary('AAPL', {
 *   modules: ['summaryProfile', 'financialData']
 * });
 * console.log(profile.summaryProfile?.longBusinessSummary);
 *
 * // Get all available modules
 * const comprehensive = await yahooFinance.quoteSummary('AAPL', {
 *   modules: 'all'
 * });
 * ```
 *
 * ### Available Modules
 *
 * See {@linkcode quoteSummary_modules} for the full list.
 *
 * ### Notes
 * * In the original **node-yahoo-finance v1**, this was incorrectly called "quote".
 *   The actual quote data is now in the {@linkcode [modules/quote] quote} module.
 *   This module provides much more comprehensive data than basic quotes.
 *
 * * **Performance**: requesting fewer modules improves response time.
 * Only request the modules you actually need.
 *
 * * **Financial Statements**: Since Nov 2024, the ~~`incomeStatementHistory*`~~,
 *   ~~`balanceSheetHistory*`~~, ~~`cashflowStatementHistory*`~~ modules provide
 *   almost no data.  Use the
 *   {@linkcode [modules/fundamentalsTimeSeries] fundamentalsTimeSeries}
 *   module instead.
 *
 * @module quoteSummary
 */
import { getTypedDefinitions } from "../lib/validate/index.js";
// @yf-schema: see the docs on how this file is automatically updated.
import optsSchema from "./quoteSummary.schema.js";
import resultsSchema from "./quoteSummary-iface.schema.js";
const optsDefinitions = getTypedDefinitions(optsSchema);
const resultsDefinitions = getTypedDefinitions(resultsSchema);
/**
 * Available quote summary modules for requesting specific data.
 *
 * Each module provides different types of financial data:
 * - **Company Info**: `assetProfile`, `summaryProfile`, `quoteType`
 * - **Price Data**: `price`, `summaryDetail`
 * - **Financial Statements**: ~~`incomeStatementHistory*`~~,
 *   ~~`balanceSheetHistory*`~~, ~~`cashflowStatementHistory*`~~. *Use
 *   {@linkcode [modules/fundamentalsTimeSeries] fundamentalsTimeSeries} instead!*
 * - **Analysis**: `recommendationTrend`, `upgradeDowngradeHistory`, `earnings*`
 * - **Ownership**: `institutionOwnership`, `fundOwnership`, `majorHoldersBreakdown`
 * - **Fund Specific**: `fundProfile`, `fundPerformance`, `topHoldings`
 * - **Other**: `calendarEvents`, `defaultKeyStatistics`, `financialData`
 *
 * Modules ending with "*" have both regular and quarterly versions.
 */
export const quoteSummary_modules = [
    "assetProfile",
    "balanceSheetHistory",
    "balanceSheetHistoryQuarterly",
    "calendarEvents",
    "cashflowStatementHistory",
    "cashflowStatementHistoryQuarterly",
    "defaultKeyStatistics",
    "earnings",
    "earningsHistory",
    "earningsTrend",
    "financialData",
    "fundOwnership",
    "fundPerformance",
    "fundProfile",
    "incomeStatementHistory",
    "incomeStatementHistoryQuarterly",
    "indexTrend",
    "industryTrend",
    "insiderHolders",
    "insiderTransactions",
    "institutionOwnership",
    "majorDirectHolders",
    "majorHoldersBreakdown",
    "netSharePurchaseActivity",
    "price",
    "quoteType",
    "recommendationTrend",
    "secFilings",
    "sectorTrend",
    "summaryDetail",
    "summaryProfile",
    "topHoldings",
    "upgradeDowngradeHistory",
];
const queryOptionsDefaults = {
    formatted: false,
    modules: ["price", "summaryDetail"],
};
/**
 * @param symbol - Stock, ETF, mutual fund, or other financial instrument symbol.
 *                 Use search() to find valid symbols.
 * @param queryOptionsOverrides - Optional configuration:
 *                                - `modules`: Specific data modules to request
 *                                - `formatted`: Return formatted strings vs raw numbers
 * @param moduleOptions - Optional module configuration (validateResult, etc.)
 *
 * @returns Promise that resolves to a QuoteSummaryResult object containing
 *          the requested modules as properties. Properties will be undefined
 *          if the module wasn't requested or isn't available for the symbol.
 *
 * @throws Will throw an error if:
 *         - Network request fails
 *         - Invalid symbol
 *         - Validation fails (if enabled)
 *
 * @see {@link QuoteSummaryOptions} for all available options
 * @see {@link QuoteSummaryResult} for result structure
 * @see {@link quoteSummary_modules} for list of all available modules
 */
export default function quoteSummary(symbol, queryOptionsOverrides, moduleOptions) {
    const financeModules = [
        "balanceSheetHistory",
        "balanceSheetHistoryQuarterly",
        "cashflowStatementHistory",
        "cashflowStatementHistoryQuarterly",
        "incomeStatementHistory",
        "incomeStatementHistoryQuarterly",
    ];
    const usedFinanceModules = financeModules.filter((m) => queryOptionsOverrides?.modules?.includes(m));
    if (usedFinanceModules.length) {
        // TODO https://github.com/gadicc/yahoo-finance2/issues/950
        const yahooFinance = this;
        yahooFinance._opts.logger.warn("QuoteSummary financial statements submodules like " +
            usedFinanceModules.join(", ") +
            " have provided almost no data since Nov 2024. Use `fundamentalsTimeSeries` instead.");
    }
    return this._moduleExec({
        moduleName: "quoteSummary",
        query: {
            assertSymbol: symbol,
            url: "https://${YF_QUERY_HOST}/v10/finance/quoteSummary/" + symbol,
            needsCrumb: true,
            definitions: optsDefinitions,
            schemaKey: "#/definitions/QuoteSummaryOptions",
            defaults: queryOptionsDefaults,
            overrides: queryOptionsOverrides,
            transformWith(options) {
                if (options.modules === "all") {
                    options.modules = quoteSummary_modules;
                }
                return options;
            },
        },
        result: {
            definitions: resultsDefinitions,
            schemaKey: "#/definitions/QuoteSummaryResult",
            // deno-lint-ignore no-explicit-any
            transformWith(result) {
                if (!result.quoteSummary) {
                    throw new Error("Unexpected result: " + JSON.stringify(result));
                }
                return result.quoteSummary.result[0];
            },
        },
        moduleOptions,
    });
}
