import type { JSONSchema } from "./validate/index.js";
import type { ModuleOptions } from "./moduleCommon.js";
type TransformFunc = (arg: any) => any;
interface ModuleExecOptions {
    /**
     * Name of the module, e.g. "search", "quoteSummary", etc.  Used in error
     * reporting.
     */
    moduleName: string;
    query: {
        /**
         * If given, a runtime assertion is performed to check that the given
         * argument is a string.  If not, a helpful error is thrown.
         */
        assertSymbol?: string;
        /**
         * URL of the API to query, WITHOUT query params.
         */
        url: string;
        /**
         * Schema definitions of which `schemaKey` below should exist.
         */
        definitions: JSONSchema["definitions"];
        /**
         * Key of schema used to validate user-provider query options.
         * e.g. yf.search('AAPL', { isThisAValidOption: "maybe" })
         */
        schemaKey: string;
        /**
         * Defaults for this query, e.g. { period: '1d' } in history,
         * and other required options that aren't often changed { locale: 'en' }.
         */
        defaults: any;
        /**
         * Query parameters generated inside the module, most commonly something
         * like { q: query } to take e.g. yf.search(query) and pass it how Yahoo
         * expects it.
         */
        runtime?: any;
        /**
         * Query options passed by the user that will override the default and
         * runtime params.  Will be validated with schemaKey.
         */
        overrides: any;
        /**
         * Called with the merged (defaults,runtime,overrides) before running
         * the query.  Useful to transform options we allow but not Yahoo, e.g.
         * allow a "2020-01-01" date but transform this to a UNIX epoch.
         */
        transformWith?: TransformFunc;
        /**
         * Default: 'json'.  Can be 'text' or 'csv' (augments fetch's "text").
         */
        fetchType?: string;
        /**
         * Default: false.  This request requires Yahoo cookies & crumb.
         */
        needsCrumb: boolean;
    };
    result: {
        /**
         * Schema definitions of which `schemaKey` below should exist.
         */
        definitions: JSONSchema["definitions"];
        /**
         * Key of schema to validate (and coerce) the retruned result from Yahoo.
         */
        schemaKey: string;
        /**
         * Mutate the Yahoo result *before* validating and coercion.  Mostly used
         * to e.g. throw if no (resault.returnField) and return result.returnField.
         */
        transformWith?: TransformFunc;
    };
    moduleOptions?: ModuleOptions;
}
type ThisWithModExec = {
    [key: string]: any;
    _moduleExec: typeof moduleExec;
};
/**
 * Executes a module with the given options.
 *
 * 1. **Query Stage**
 *     1. Validate user-supplied module params, e.g. { period: '1d' }
 *     1. Merge query params: (module defaults, user-supplied overrides, etc)
 *     1. Call `transformWith` func if provided.
 *
 * 1. Call `lib/yahooFinanceFetch.ts` with the above.
 *
 * 1. **Result Stage**
 *     1. Call `transformWith` func if provided.
 *     1. Validate the result and coerce types.
 *
 * @param this - The context of the module execution.
 * @param opts - The options for the module execution.
 * @returns A promise that resolves to the result of the module execution.
 *   It will be validated according to the requested schema but the calling
 *   module should cast to the correct type.
 */
declare function moduleExec(this: ThisWithModExec, opts: ModuleExecOptions): Promise<unknown>;
export default moduleExec;
//# sourceMappingURL=moduleExec.d.ts.map