/**
 * The "root" store class.  Unless you are implementing a new store, you
 * probably want to use one of the existing sub-classe, such as `fs` or
 * `memory`.
 *
 * @module
 */
import type { FMCCacheContent } from "./cache.js";
import type { FetchCacheOptions, Runtime } from "./fetch-cache.js";
/**
 * The "root" options for any store.  You can extend these with store-specific
 * options in sub-classes.  Note, the "required" `runtime` parameter is
 * automatically passed to the store by `createFetchCache`.
 */
export interface FMCStoreOptions {
    runtime: Runtime;
}
/**
 * The "root" store class.  Don't instantiate directly, either use an existing
 * sub-class (e.g. `fs` or `memory`) or extend this class to create a new store
 * class, overriding at least `fetchContent` and `storeContent`, and perhaps,
 * `idFromRequest`, the constructor and others according to your needs.
 *
 * @example
 * ```ts
 * import FMCStore from "fetch-mock-cache/store";
 * import type { FMCCacheContent, FMCStoreOptions } from "fetch-mock-cache/store";
 *
 * export default class MyStore extends FMCStore {
 *   async fetchContent(req: FMCCacheContent["request"]) {
 *     // your implementation here
 *   }
 *   async storeContent(content: FMCCacheContent) {
 *    // your implementation here
 *   }
 * }
 * ```
 */
export default class FMCStore {
    runtime: Runtime;
    constructor(options: FMCStoreOptions);
    /**
     * Given an input string, return a SHA-256 hash of the string, truncated to
     * 7 characters, or to `length`, if specified.
     * @param input - the input string
     * @param length - truncated length of the hash, default: 7
     * @returns the hash, truncated to 7 characters, or to `length`, if specified.
     */
    hash(input: string, length?: number): Promise<string>;
    /**
     * Given an FMCCacheContent.request object, return a Record<string,string>
     * of indentifiers that are unique to the request (e.g. method, headers, body).
     * This will later be used by `idFromRequest` to generate a single unique
     * identifier string.  For each new request, the store will check if a cached
     * copy with the id exists, in which case, it will be returned instead, rather
     * than performing an actual network request.
     * @param request an FMCCacheContent.request object
     * @param hashLen default hash length to use, default: 7
     * @returns a Record<string,string> of unique identifiers, or null if none
     */
    uniqueRequestIdentifiers(request: FMCCacheContent["request"], hashLen?: number): Promise<Record<string, string> | null>;
    /**
     * Given an FMCCacheContent.request object, return a unique string id to
     * identify it.  Uses `uniqueRequestIdentifiers` under the hood.
     * @param request An FMCCacheContent.request object
     * @returns A unique string id
     */
    idFromRequest(request: FMCCacheContent["request"], options?: FetchCacheOptions): Promise<string>;
    /**
     * Given an FMCCacheContent.request object, return the cached content
     * from the store if it exists, or undefined otherwise.
     * @param {FMCCacheContent["request"]} req
     * @returns {Promise<FMCCacheContent | undefined>}
     */
    fetchContent(request: FMCCacheContent["request"], options?: FetchCacheOptions): Promise<FMCCacheContent | null | undefined>;
    /**
     * Given an FMCCacheContent object, store it in the store.
     * @param {FMCCacheContent} content
     */
    storeContent(content: FMCCacheContent, options?: FetchCacheOptions): Promise<void>;
}
export type { FMCCacheContent };
